#include "gpib.h"
#include "formatio.h"
#include "tg1304.h"

/*= Thurlby Thandar TG1304 13MHz Function Generator =======================*/
/* LabWindows 2.0 Multi-Instrument Driver                                            */
/* Original Release: Nov 1994                                                       */
/* By: John Tothill, Thurlby Thandar Instruments Ltd.                                                                     */
/* Originally written in C                                                 */
/* Modification History: None                                              */
/*=========================================================================*/

/*= INSTRUMENT TABLE ======================================================*/
/* address array: contains the GPIB addresses of opened instruments.       */
/* bd array: contains the device descriptors returned by OpenDev.          */
/* instr_cnt: contains the number of instruments open of this model type.  */
/* tg1304_err: the error variable for the instrument module                */
/*=========================================================================*/
static int address[TG1304_MAX_INSTR + 1];
static int bd[TG1304_MAX_INSTR + 1];
static int instr_cnt;
static int tg1304_err;
static int tg1304_clear[TG1304_MAX_INSTR + 1];
                                     /* holds trigger/gate mode from reset */

/*= INTERNAL DATA =========================================================*/
/* cmd is a buffer for GPIB I/O strings */
static char cmd[100];

/*= INSTRUMENT-DEPENDENT COMMAND ARRAYS ===================================*/
static char *main_func[8];
static char *main_out[2];
static char *aux_func[6];
static char *aux_out[2];
static char *mod_enab[3];
static char *mod_wave[4];
static char *vcaf_mode[4];
static char *set_status[3];
static char *set_store[2];
static char *get_status[5];
/*=========================================================================*/

/*= UTILITY ROUTINES ======================================================*/
int tg1304_open_instr (int);
int tg1304_close_instr (int);
int tg1304_invalid_integer_range (int, int, int, int);
int tg1304_invalid_longint_range (long, long, long, int);
int tg1304_invalid_real_range (double, double, double, int);
int tg1304_device_closed (int);
int tg1304_read_data (int, char *, int);
int tg1304_write_data (int, char *, int);
int tg1304_read_data_file (int, char *);
int tg1304_write_data_file (int, char *);
void tg1304_setup_arrays (void);
/*=========================================================================*/
/* Function: Initialize                                                    */
/* Purpose:  This function opens the instrument, queries the instrument    */
/*           for its ID, and initializes the instrument to a known state.  */
/*=========================================================================*/
int tg1304_init (addr, id_query, rest, instrID)
int addr;
int id_query;
int rest;
int *instrID;
{
int  ID;

    if (tg1304_invalid_integer_range (addr, 0, 30, -1) != 0)
        return tg1304_err;
    if (tg1304_invalid_integer_range (id_query, 0, 1, -2) != 0)
        return tg1304_err;
    if (tg1304_invalid_integer_range (rest, 0, 5, -3) != 0)
        return tg1304_err;

    ID = tg1304_open_instr (addr);
    if (ID <= 0)
        return tg1304_err;

    if (id_query)
        {
        if (tg1304_write_data (ID, "*IDN?\012", 6) != 0)
            {
            tg1304_close_instr (ID);
            return tg1304_err;
            }
        if (tg1304_read_data (ID, cmd, 50) != 0)
            {
            tg1304_close_instr (ID);
            return tg1304_err;
            }

        Scan (cmd, "TTI,TG1304");
        if (NumFmtdBytes () != 10)
            {
            tg1304_err = 223;
            tg1304_close_instr (ID);
            return tg1304_err;
            }
        }

    if (rest)
        {
        if (tg1304_reset(ID,rest))
            {
            tg1304_clear[ID] = 0;
            return(tg1304_err);
            }
         }
    else
        tg1304_clear[ID] = 0;

    tg1304_setup_arrays ();
    *instrID = ID;
    return tg1304_err;
}
/*=========================================================================*/
/* Function: tg1304_config_main                                            */
/* Purpose:  Configure main generator waveform, frequency, output level and*/
/*           output state                                                  */
/*=========================================================================*/
int tg1304_config_main (instrID, func, output, freq, level)
int instrID;
int func;
int output;
double freq;
double level;
{
    if (tg1304_invalid_integer_range (instrID, 1, TG1304_MAX_INSTR, -1) != 0)
        return tg1304_err;
    if (tg1304_invalid_integer_range (func, 0, 7, -2) != 0)
        return tg1304_err;
    if (tg1304_invalid_integer_range (output, 0, 1, -3) != 0)
        return tg1304_err;
    if (tg1304_invalid_real_range (freq, 1e-2, 13e6, -4) != 0)
        return tg1304_err;

    if (tg1304_device_closed (instrID))
        return tg1304_err;

    Fmt(cmd,"MAIN %s;MAIN %s;MFRQ %f;MVPP %f\012"
                                ,main_func[func],main_out[output],freq,level);
    if (tg1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tg1304_err;

    return tg1304_err;
}
/*=========================================================================*/
/* Function: tg1304_main_offset                                            */
/* Purpose:  set the main generator dc offset                              */
/*=========================================================================*/
void tg1304_main_offset (instrID, offset)
int instrID;
double offset;
{
    if (tg1304_invalid_integer_range (instrID, 1, TG1304_MAX_INSTR, -1) != 0)
        return;
    if (tg1304_invalid_real_range (offset, -10.0, 10.0, -5) != 0)
        return;

    if (tg1304_device_closed (instrID))
        return;

    Fmt (cmd, "MDCO %f\012", offset);
    if (tg1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return;

}
/*=========================================================================*/
/* Function: tg1304_main_symmetry                                          */
/* Purpose:  Set the main generator symmetry                               */
/*=========================================================================*/
void tg1304_main_symmetry (instrID, symm)
int instrID;
double symm;
{
    if (tg1304_invalid_integer_range (instrID, 1, TG1304_MAX_INSTR, -1) != 0)
        return;
    if (tg1304_invalid_real_range (symm, 5.0, 95.0, -6) != 0)
        return;

    if (tg1304_device_closed (instrID))
        return;

    Fmt (cmd, "MSYM %f\012", symm);
    if (tg1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return;

}
/*=========================================================================*/
/* Function: tg1304_main_phase                                             */
/* Purpose:  Set the main generator phase                                  */
/*=========================================================================*/
void tg1304_main_phase (instrID, phase)
int instrID;
int phase;
{
    if (tg1304_invalid_integer_range (instrID, 1, TG1304_MAX_INSTR, -1) != 0)
        return;
    if (tg1304_invalid_integer_range (phase, -360, 360, -7) != 0)
        return;

    if (tg1304_device_closed (instrID))
        return;

    Fmt (cmd, "MPHZ %d", phase);
    if (tg1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return;

}
/*=========================================================================*/
/* Function: tg1304_config aux                                             */
/* Purpose:  Configure aux generator waveform, frequency, output level and*/
/*           output state                                                  */
/*=========================================================================*/
int tg1304_config_aux (instrID, func, output, freq, level)
int instrID;
int func;
int output;
double freq;
double level;
{
    if (tg1304_invalid_integer_range (instrID, 1, TG1304_MAX_INSTR, -1) != 0)
        return tg1304_err;
    if (tg1304_invalid_integer_range (func, 0, 5, -2) != 0)
        return tg1304_err;
    if (tg1304_invalid_integer_range (output, 0, 1, -3) != 0)
        return tg1304_err;
    if (tg1304_invalid_real_range (freq, 5e-3, 50e3, -4) != 0)
        return tg1304_err;

    if (tg1304_device_closed (instrID))
        return tg1304_err;

    Fmt(cmd,"AUX %s;AUX %s;XFRQ %f;XVPP %f\012"
                                ,aux_func[func],aux_out[output],freq,level);
    if (tg1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tg1304_err;

    return tg1304_err;
}
/*=========================================================================*/
/* Function: tg1304_aux_symmetry                                           */
/* Purpose:  Set the aux generator symmetry                               */
/*=========================================================================*/
void tg1304_aux_symmetry (instrID, symm)
int instrID;
int symm;
{
    if (tg1304_invalid_integer_range (instrID, 1, TG1304_MAX_INSTR, -1) != 0)
        return;
    if (tg1304_invalid_integer_range (symm, 5, 95, -5) != 0)
        return;

    if (tg1304_device_closed (instrID))
        return;

    Fmt (cmd, "XSYM %d\012", symm);
    if (tg1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return;

}
/*=========================================================================*/
/* Function: tg1304_config_am                                              */
/* Purpose:  Configure amplitude modulation                                */
/*=========================================================================*/
int tg1304_config_am (instrID, enable, wave, depth, freq)
int instrID;
int enable;
int wave;
int depth;
double freq;
{
    if (tg1304_invalid_integer_range (instrID, 1, TG1304_MAX_INSTR, -1) != 0)
        return tg1304_err;
    if (tg1304_invalid_integer_range (enable, 0, 2, -2) != 0)
        return tg1304_err;
    if (tg1304_invalid_integer_range (wave, 0, 3, -3) != 0)
        return tg1304_err;
    if (tg1304_invalid_integer_range (depth, 1, 99, -4) != 0)
        return tg1304_err;
    if (tg1304_invalid_real_range (freq, 5e-3, 50e3, -5) != 0)
        return tg1304_err;

    if (tg1304_device_closed (instrID))
        return tg1304_err;

    if (enable == 0) /* on so send complete command*/
        {
        if (wave == 3) /* external */
            Fmt (cmd,"AM %s; AM %s;AMDP %d\012"
                            ,mod_enab[enable],mod_wave[wave],depth);
        else
            Fmt (cmd, "AM %s;AM %s;AMDP %d;XFRQ %f\012"
                                ,mod_enab[enable],mod_wave[wave],depth,freq);
        }
    else /* off or clear so send only enable */
        Fmt (cmd,"AM %s\012",mod_enab[enable]);
    if (tg1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tg1304_err;

    return tg1304_err;
}
/*=========================================================================*/
/* Function: tg1304_config_vca                                             */
/* Purpose:  Configure voltage controlled attenuetion                      */
/*=========================================================================*/
int tg1304_config_vca (instrID, enable, vcamode, time, minampl, maxampl)
int instrID;
int enable;
int vcamode;
double time;
double minampl;
double maxampl;
{
    if (tg1304_invalid_integer_range (instrID, 1, TG1304_MAX_INSTR, -1) != 0)
        return tg1304_err;
    if (tg1304_invalid_integer_range (enable, 0, 2, -2) != 0)
        return tg1304_err;
    if (tg1304_invalid_integer_range (vcamode, 0, 3, -3) != 0)
        return tg1304_err;
    if (tg1304_invalid_real_range (time, 20e-6, 1e3, -4) != 0)
        return tg1304_err;

    if (tg1304_device_closed (instrID))
        return tg1304_err;

    if (enable == 0) /* on so send complete command*/
        {
        if (vcamode > 1) /* external */
            Fmt (cmd,"VCA %s; VCA %s;VSTA %f;VSTP %f\012"
                        ,mod_enab[enable],vcaf_mode[vcamode],minampl,maxampl);
        else
            Fmt (cmd, "VCA %s;VCA %s;VSTA %f;VSTP %f;XPER %f\012"
                   ,mod_enab[enable],vcaf_mode[vcamode],minampl,maxampl,time);
        }
    else /* off or clear so send only enable */
        Fmt (cmd,"VCA %s\012",mod_enab[enable]);
    if (tg1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tg1304_err;

    return tg1304_err;
}
/*=========================================================================*/
/* Function: tg1304_config_sum                                             */
/* Purpose:  Configure the summing function                                */
/*=========================================================================*/
int tg1304_config_sum (instrID, enable, wave, ampl, freq)
int instrID;
int enable;
int wave;
double ampl;
double freq;
{
    if (tg1304_invalid_integer_range (instrID, 1, TG1304_MAX_INSTR, -1) != 0)
        return tg1304_err;
    if (tg1304_invalid_integer_range (enable, 0, 2, -2) != 0)
        return tg1304_err;
    if (tg1304_invalid_integer_range (wave, 0, 3, -3) != 0)
        return tg1304_err;
    if (tg1304_invalid_real_range (freq, 5e-3, 50e3, -4) != 0)
        return tg1304_err;

    if (tg1304_device_closed (instrID))
        return tg1304_err;

    if (enable == 0) /* on so send complete command*/
        {
        if (wave == 3) /* external */
            Fmt (cmd,"SUM %s; SUM %s;SVPP %f\012"
                            ,mod_enab[enable],mod_wave[wave],ampl);
        else
            Fmt (cmd, "SUM %s;SUM %s;SVPP %f;XFRQ %f\012"
                                ,mod_enab[enable],mod_wave[wave],ampl,freq);
        }
    else /* off or clear so send only enable */
        Fmt (cmd,"SUM %s\012",mod_enab[enable]);
    if (tg1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tg1304_err;

    return tg1304_err;
}
/*=========================================================================*/
/* Function: tg1304_config_fm                                              */
/* Purpose:  Configure frequency modulation                                */
/*=========================================================================*/
int tg1304_config_fm (instrID, enable, wave, dev, freq)
int instrID;
int enable;
int wave;
double dev;
double freq;
{
    if (tg1304_invalid_integer_range (instrID, 1, TG1304_MAX_INSTR, -1) != 0)
        return tg1304_err;
    if (tg1304_invalid_integer_range (enable, 0, 2, -2) != 0)
        return tg1304_err;
    if (tg1304_invalid_integer_range (wave, 0, 3, -3) != 0)
        return tg1304_err;
    if (tg1304_invalid_real_range (freq, 5e-3, 50e3, -4) != 0)
        return tg1304_err;

    if (tg1304_device_closed (instrID))
        return tg1304_err;

    if (enable == 0) /* on so send complete command*/
        {
        if (wave == 3) /* external */
            Fmt (cmd,"FM %s; FM %s;FMDV %f\012"
                            ,mod_enab[enable],mod_wave[wave],dev);
        else
            Fmt (cmd, "FM %s;FM %s;FMDV %f;XFRQ %f\012"
                                ,mod_enab[enable],mod_wave[wave],dev,freq);
        }
    else /* off or clear so send only enable */
        Fmt (cmd,"FM %s\012",mod_enab[enable]);
    if (tg1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tg1304_err;

    return tg1304_err;
}
/*=========================================================================*/
/* Function: tg1304_config_sweep                                           */
/* Purpose:  Configure sweep                                               */
/*=========================================================================*/
int tg1304_config_sweep (instrID, enable, vcfmode, time, startf, stopf)
int instrID;
int enable;
int vcfmode;
double time;
double startf;
double stopf;
{
    if (tg1304_invalid_integer_range (instrID, 1, TG1304_MAX_INSTR, -1) != 0)
        return tg1304_err;
    if (tg1304_invalid_integer_range (enable, 0, 2, -2) != 0)
        return tg1304_err;
    if (tg1304_invalid_integer_range (vcfmode, 0, 3, -3) != 0)
        return tg1304_err;
    if (tg1304_invalid_real_range (time, 20e-6, 1e3, -4) != 0)
        return tg1304_err;
    if (tg1304_invalid_real_range (startf, 1e-2, 13e6, -5) != 0)
        return tg1304_err;
    if (tg1304_invalid_real_range (stopf, 1e-2, 13e6, -6) != 0)
        return tg1304_err;

    if (tg1304_device_closed (instrID))
        return tg1304_err;

    if (enable == 0) /* on so send complete command*/
        {
        if (vcfmode > 1) /* external */
            Fmt (cmd,"VCF %s; VCF %s\012"
                        ,mod_enab[enable],vcaf_mode[vcfmode]);
        else
            Fmt (cmd, "VCF %s;VCF %s;FSTA %f;FSTP %f;XPER %f\012"
                   ,mod_enab[enable],vcaf_mode[vcfmode],startf,stopf,time);
        }
    else /* off or clear so send only enable */
        Fmt (cmd,"VCF %s\012",mod_enab[enable]);
    if (tg1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tg1304_err;

    return tg1304_err;
}
/*=========================================================================*/
/* Function: tg1304_trig                                                   */
/* Purpose:  Gives access to the instrumnets trigger and gate capabilities.*/
/*           NB the instrument MUST have been reset to the required mode   */
/*           using the RESET function provided in this driver otherwise    */
/*           this function will fail and return an error.                  */
/*=========================================================================*/
int tg1304_trig (instrID, enab, gont, gofft, thrshld, trgper, burst)
int instrID;
int enab;
double gont;
double gofft;
double thrshld;
double trgper;
double burst;
{
    if (tg1304_invalid_integer_range (instrID, 1, TG1304_MAX_INSTR, -1) != 0)
        return tg1304_err;
    if (tg1304_invalid_integer_range (enab, 0, 1, -2) != 0)
        return tg1304_err;
    if (tg1304_device_closed (instrID))
        return tg1304_err;

    switch (tg1304_clear[instrID])
    {
    case 0: /* not reset */
        tg1304_err = -10;
        return tg1304_err;
        break;
    case 1: /* trigger not configured */
        tg1304_err = -11;
        return tg1304_err;
        break;
    }

    if (enab == 1) /* off */
        {
        Fmt (cmd, "TRIG OFF\012");
        if (tg1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
            return tg1304_err;
        return tg1304_err;
        }

    if (tg1304_invalid_real_range (thrshld, -10.0, 10.0, -5) != 0)
        return tg1304_err;

    switch (tg1304_clear[instrID])
    {
    case 2: /* burst int */
    case 3: /* burst ext */
        if (tg1304_invalid_real_range (trgper,150e-9,850, -6) != 0)
            return tg1304_err;
        if (tg1304_invalid_real_range (burst,0.5,524288.5, -7) != 0)
            return tg1304_err;
        if (tg1304_clear[instrID] == 2) /* internal */
            {
            Fmt (cmd, "TRIG ON;BCNT %f;TPER %f\012",burst,trgper);
            if (tg1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
                return tg1304_err;
            return tg1304_err;
            }
         else  /* external */
            {
            Fmt (cmd, "TRIG ON;BCNT %f;TRGV %f\012",burst,thrshld);
            if (tg1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
                return tg1304_err;
            return tg1304_err;
            }
        break;
    case 4: /* gate int */
    case 5: /* gate ext */
        if (tg1304_invalid_real_range (gont,800e-9,850, -3) != 0)
            return tg1304_err;
        if (tg1304_invalid_real_range (gofft,800e-9,850, -4) != 0)
            return tg1304_err;
        if (tg1304_clear[instrID] == 4) /* internal */
            {
            Fmt (cmd, "TRIG ON;GON %f;GOFF %f\012",gont,gofft);
            if (tg1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
                return tg1304_err;
            return tg1304_err;
            }
         else  /* external */
            {
            Fmt (cmd, "TRIG ON;TRGV %f\012",thrshld);
            if (tg1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
                return tg1304_err;
            return tg1304_err;
            }
        break;
    }
    return(0);
}
/*=========================================================================*/
/* Function: tg1304_set_status                                             */
/* Purpose:  Set the value in the Service Request Enable, Event Status     */
/*           Enable or Parallel Poll Enable register                       */
/*=========================================================================*/
int tg1304_set_status (instrID, reg, val)
int instrID;
int reg;
int val;
{
    if (tg1304_invalid_integer_range (instrID, 1, TG1304_MAX_INSTR, -1) != 0)
        return tg1304_err;
    if (tg1304_invalid_integer_range (reg, 0, 2, -2) != 0)
        return tg1304_err;
    if (tg1304_invalid_integer_range (val, 0, 255, -3) != 0)
        return tg1304_err;

    if (tg1304_device_closed (instrID))
        return tg1304_err;

    Fmt (cmd, "%s %d\012", set_status[reg],val);
    if (tg1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tg1304_err;

    return tg1304_err;
}
/*=========================================================================*/
/* Function: tg1304_loc_stores                                             */
/* Purpose:  Allows saving to or recalling from the instruments local non- */
/*           volatile stores                                               */
/*=========================================================================*/
int tg1304_loc_stores (instrID, op, storeno)
int instrID;
int op;
int storeno;
{
    if (tg1304_invalid_integer_range (instrID, 1, TG1304_MAX_INSTR, -1) != 0)
        return tg1304_err;
    if (tg1304_invalid_integer_range (op, 0, 1, -1) != 0)
        return tg1304_err;
    if (tg1304_invalid_integer_range (storeno, 1, 64, -2) != 0)
        return tg1304_err;

    if (tg1304_device_closed (instrID))
        return tg1304_err;

    Fmt (cmd, "%s %d\012", set_store[op],storeno);
    if (tg1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tg1304_err;

    return tg1304_err;
}
/*=========================================================================*/
/* Function: tg1304_get status                                             */
/* Purpose:  Reads the value in the selected iEEE 488.2 status register    */
/*=========================================================================*/
int tg1304_get_status (instrID, reg, val)
int instrID;
int reg;
int *val;
{
    if (tg1304_invalid_integer_range (instrID, 1, TG1304_MAX_INSTR, -1) != 0)
        return tg1304_err;
    if (tg1304_invalid_integer_range (reg, 0, 4, -2) != 0)
        return tg1304_err;

    if (tg1304_device_closed (instrID))
        return tg1304_err;

    Fmt (cmd, "%s\012", get_status[reg]);
    if (tg1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tg1304_err;
    if (tg1304_read_data (instrID, cmd, 50) != 0)
        return tg1304_err;
    if (Scan (cmd, "%d",val) != 1)
        tg1304_err = 236;

    return tg1304_err;
}
/*=========================================================================*/
/* Function: tg1304_snd_cmd                                                */
/* Purpose:  Sends the data in the supplied buffer to the instrument       */
/*           Use for commands not available in this driver - e.g. inc and  */
/*           dec commands.												   */
/*=========================================================================*/
int tg1304_snd_cmd (instrID, command)
int instrID;
char *command;
{
char output_buf[200];
    if (tg1304_invalid_integer_range (instrID, 1, TG1304_MAX_INSTR, -1) != 0)
        return tg1304_err;

    if (tg1304_device_closed (instrID))
        return tg1304_err;

    Fmt(output_buf,"%s",command);
    if (tg1304_write_data(instrID,command,NumFmtdBytes()) != 0)
        return tg1304_err;

    return tg1304_err;
}
/*=========================================================================*/
/* Function: tg1304_get_response                                           */
/* Purpose:  Gets a response to a previous query. Used in conjuction with  */
/*           the send command function                                     */
/*=========================================================================*/
int tg1304_get_resp (instrID, response, nbytes)
int instrID;
char *response;
int *nbytes;
{
    if (tg1304_invalid_integer_range (instrID, 1, TG1304_MAX_INSTR, -1) != 0)
        return tg1304_err;

    if (tg1304_device_closed (instrID))
        return tg1304_err;

    if (tg1304_read_data(instrID,response,2000))
        return tg1304_err;
    *nbytes = ibcnt;

    return tg1304_err;
}
/*=========================================================================*/
/* Function: tg1304_file_ops                                               */
/* Purpose:  Send or receive files                                         */
/*=========================================================================*/
int tg1304_file_ops (instrID, op, fname)
int instrID;
int op;
char *fname;
{
    if (tg1304_invalid_integer_range (instrID, 1, TG1304_MAX_INSTR, -1) != 0)
        return tg1304_err;
    if (tg1304_invalid_integer_range (op, 0, 1, -2) != 0)
        return tg1304_err;

    if (tg1304_device_closed (instrID))
        return tg1304_err;

    if (op == 1) /* get to file from tg1304 */
        {
        if (tg1304_read_data_file(instrID,fname))
            return tg1304_err;
        }
    else /* send from file to tg1304 */
        if (tg1304_write_data_file(instrID,fname))
            return tg1304_err;

    return tg1304_err;
}
/*=========================================================================*/
/* Function: tg1304_reset                                                  */
/* Purpose:  Resets the instrument by sending a setup binary block to store*/
/*           64 then recalling  it                                         */
/*=========================================================================*/
int tg1304_reset(instrID,rst)
int instrID;
int rst;
{
char s[390];
int i;
int n;
    tg1304_clear[instrID] = 0;

    if (tg1304_invalid_integer_range (instrID, 1, TG1304_MAX_INSTR, -1) != 0)
        return tg1304_err;
    if (tg1304_invalid_integer_range (rst, 1, 5, -3) != 0)
        return tg1304_err;
    if (tg1304_device_closed (instrID))
        return tg1304_err;

    i = 0;
    n = 21;
    switch (rst)
    {
    case 1: /* trigger clear */
CopyBytes(s,i,"\x053\x054\x04F\x052\x045\x020\x036\x034\x02C\x023\x033\x033\x037\x033\x001\x000\x000\x000\x000\x002\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x002\x000\x000\x000\x002\x000\x000\x010\x004\x000\x001\x000\x004\x00A\x003\x002\x002",0,n);
i = i + 21;
CopyBytes(s,i,"\x003\x000\x000\x000\x000\x000\x000\x000\x08E\x09C\x040\x000\x000\x000\x000\x000\x080\x080\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x080\x080\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x08A\x0FA\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x080\x080\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x081\x080\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x0B4\x00A",0,10);
        break;
    case 2: /* burst int */
CopyBytes(s,i,"\x053\x054\x04F\x052\x045\x020\x036\x034\x02C\x023\x033\x033\x037\x033\x002\x000\x000\x000\x000\x00C\x00C",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x046\x000\x000\x002\x000\x000\x000\x002\x000\x000\x010\x004\x000\x001\x000\x004\x00A\x003\x002\x002",0,n);
i = i + 21;
CopyBytes(s,i,"\x003\x000\x000\x000\x000\x000\x000\x000\x08E\x09C\x040\x000\x000\x000\x000\x000\x080\x080\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x080\x080\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x08A\x0FA\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x080\x080\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x081\x080\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x07D\x0CC\x0CC\x0CC\x0CC\x0CC",0,n);
i = i + 21;
CopyBytes(s,i,"\x0CC\x0CD\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x001\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x024\x00A",0,10);
        break;
    case 3: /* burst ext */
CopyBytes(s,i,"\x053\x054\x04F\x052\x045\x020\x036\x034\x02C\x023\x033\x033\x037\x033\x002\x000\x000\x000\x000\x007\x007",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x054\x000\x000\x002\x000\x000\x000\x002\x000\x000\x010\x004\x000\x001\x000\x004\x00A\x003\x002\x002",0,n);
i = i + 21;
CopyBytes(s,i,"\x003\x000\x000\x000\x000\x000\x000\x000\x08E\x09C\x040\x000\x000\x000\x000\x000\x080\x080\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x080\x080\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x08A\x0FA\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x080\x080\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x081\x080\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x001\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x016\x00A",0,10);
        break;
    case 4: /* gate int */
CopyBytes(s,i,"\x053\x054\x04F\x052\x045\x020\x036\x034\x02C\x023\x033\x033\x037\x033\x002\x000\x000\x000\x000\x012\x012",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x063\x000\x000\x002\x000\x000\x000\x002\x000\x000\x010\x004\x000\x001\x000\x004\x00A\x003\x002\x002",0,n);
i = i + 21;
CopyBytes(s,i,"\x003\x000\x000\x000\x000\x000\x000\x000\x08E\x09C\x040\x000\x000\x000\x000\x000\x080\x080\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x080\x080\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x08A\x0FA\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x080\x080\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x081\x080\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x07D\x0CC\x0CC\x0CC\x0CC\x0CC",0,n);
i = i + 21;
CopyBytes(s,i,"\x0CC\x0CD\x07C\x0CC\x0CC\x0CC\x0CC\x0CC\x0CC\x0CD\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x05D\x00A",0,10);
        break;
    case 5: /* gate ext */
CopyBytes(s,i,"\x053\x054\x04F\x052\x045\x020\x036\x034\x02C\x023\x033\x033\x037\x033\x002\x000\x000\x000\x000\x00F\x00F",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x070\x000\x000\x002\x000\x000\x000\x002\x000\x000\x010\x004\x000\x001\x000\x004\x00A\x003\x002\x002",0,n);
i = i + 21;
CopyBytes(s,i,"\x003\x000\x000\x000\x000\x000\x000\x000\x08E\x09C\x040\x000\x000\x000\x000\x000\x080\x080\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x080\x080\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x08A\x0FA\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x080\x080\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x081\x080\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000\x000",0,n);
i = i + 21;
CopyBytes(s,i,"\x000\x000\x000\x000\x000\x000\x000\x000\x041\x00A",0,10);
        break;
    }
    if (tg1304_write_data (instrID, s, 388) != 0)
            return tg1304_err;
    /* recall store 64 */
    if (tg1304_write_data (instrID, "*RCL 64\012", 8) != 0)
            return tg1304_err;

    Fmt (cmd, "TRIG OFF\012");
    if (tg1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tg1304_err;

    tg1304_clear[instrID] = rst;
    return(tg1304_err);
}
/*=========================================================================*/
/* Function: Close                                                         */
/* Purpose:  This function closes the instrument.                          */
/*=========================================================================*/
int tg1304_close (instrID)
int instrID;
{
    if (tg1304_invalid_integer_range (instrID, 1, TG1304_MAX_INSTR, -1) != 0)
        return tg1304_err;

    if (tg1304_device_closed (instrID))
        return tg1304_err;

    tg1304_close_instr (instrID);
    return tg1304_err;
}

/*= UTILITY ROUTINES ======================================================*/

/*=========================================================================*/
/* Function: Open Instrument                                               */
/* Purpose:  This function locates and initializes an entry in the         */
/*           Instrument Table and the GPIB device table for the            */
/*           instrument.  If successful, the instrument ID is returned,    */
/*           else a -1 is returned.  The size of the Instrument Table can  */
/*           be changed in the include file by altering the constant       */
/*           TG1304_MAX_INSTR.                                             */
/*=========================================================================*/
int tg1304_open_instr (addr)
int addr;
{
    int i, instrID;

    instrID = 0;

/* Check to see if the instrument is already in the Instrument Table. */

    for (i=1; i<= TG1304_MAX_INSTR; i++)
        if (address[i] == addr)
            {
            instrID = i;
            i = TG1304_MAX_INSTR;
            }

/* If it is not in the instrument table, open an entry for the instrument. */

    if (instrID <= 0)
        for (i=1; i<= TG1304_MAX_INSTR; i++)
            if (address[i] == 0)
                {
                instrID = i;
                i = TG1304_MAX_INSTR;
                }

/* If an entry could not be opened in the Instrument Table, return an error.*/

    if (instrID <= 0)
        {
        tg1304_err = 220;
        return -1;
        }

/*  If the device has not been opened in the GPIB device table (bd[ID] = 0),*/
/*  then open it.                                                           */

    if (bd[instrID] <= 0)
        {
        if (instr_cnt <= 0)
            CloseInstrDevs("TG1304");
        bd[instrID] = OpenDev ("", "TG1304");
        if (bd[instrID] <= 0)
            {
            tg1304_err = 220;
            return -1;
            }
        instr_cnt += 1;
        address[instrID] = addr;
        }

/*  Change the primary address of the device    */

    if (ibpad (bd[instrID], addr) < 0)
        {
        tg1304_err = 233;
        return -1;
        }

    return instrID;
}
/*=========================================================================*/
/* Function: Close Instrument                                              */
/* Purpose:  This function closes the instrument by removing it from the   */
/*           GPIB device table and setting the address and the bd to zero  */
/*           in the Instrument Table.  The return value is equal to the    */
/*           global error variable.                                        */
/*=========================================================================*/
int tg1304_close_instr (instrID)
int instrID;
{
    if (bd[instrID] != 0)
        {
        CloseDev (bd[instrID]);
        bd[instrID] = 0;
        address[instrID] = 0;
        instr_cnt -= 1;
        }
    else
        tg1304_err = 221;

    return tg1304_err;
}
/*=========================================================================*/
/* Function: Invalid Integer Range                                         */
/* Purpose:  This function checks an integer to see if it lies between a   */
/*           minimum and maximum value.  If the value is out of range, set */
/*           the global error variable to the value err_code.  If the      */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int tg1304_invalid_integer_range (val, min, max, err_code)
int val;
int min;
int max;
int err_code;
{
  if ((val < min) || (val > max))
    {
    tg1304_err = err_code;
    return -1;
    }
  return 0;
}
/*=========================================================================*/
/* Function: Invalid Long Integer Range                                    */
/* Purpose:  This function checks a long integer to see if it lies between */
/*           a minimum and maximum value.  If the value is out of range,   */
/*           set the global error variable to the value err_code.  If the  */
/*           value is OK, error = 0.  The return value is equal to the     */
/*           global error value.                                           */
/*=========================================================================*/
int tg1304_invalid_longint_range (val, min, max, err_code)
long val;
long min;
long max;
int  err_code;
{
    if (val < min || val > max)
        {
        tg1304_err = err_code;
        return -1;
        }
    return 0;
}
/*=========================================================================*/
/* Function: Invalid Real Range                                            */
/* Purpose:  This function checks a real number to see if it lies between  */
/*           a minimum and maximum value.  If the value is out of range,   */
/*           set the global error variable to the value err_code.  If the  */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int tg1304_invalid_real_range (val, min, max, err_code)
double val;
double min;
double max;
int err_code;
{
  if ((val < min) || (val > max))
    {
    tg1304_err = err_code;
    return -1;
    }
  return 0;
}
/*=========================================================================*/
/* Function: Device Closed                                                 */
/* Purpose:  This function checks to see if the module has been            */
/*           initialized.  If the device has not been opened, a 1 is       */
/*           returned, 0 otherwise.                                        */
/*=========================================================================*/
int tg1304_device_closed (instrID)
int instrID;
{
    if (bd[instrID] <= 0)
        {
        tg1304_err = 232;
        return -1;
        }
    return 0;
}
/*=========================================================================*/
/* Function: Read Data                                                     */
/* Purpose:  This function reads a buffer of data from the instrument. The */
/*           return value is equal to the global error variable.           */
/*=========================================================================*/
int tg1304_read_data (instrID, buf, cnt)
int instrID;
char *buf;
int cnt;
{
    if (ibrd(bd[instrID], buf, (long)cnt) <= 0)
        tg1304_err = 231;
    else
        tg1304_err = 0;

    return tg1304_err;
}
/*=========================================================================*/
/* Function: Write Data                                                    */
/* Purpose:  This function writes a buffer of data to the instrument. The  */
/*           return value is equal to the global error variable.           */
/*=========================================================================*/
int tg1304_write_data (instrID, buf, cnt)
int instrID;
char *buf;
int cnt;
{
    if (ibwrt(bd[instrID], buf, (long)cnt) <= 0)
        tg1304_err = 230;
    else
        tg1304_err = 0;

    return tg1304_err;
}
/*=========================================================================*/
/* Function: Read Data File                                                */
/* Purpose:  This function reads a buffer of data from the instrument and  */
/*           stores it to the file specified by "filename".  Filename must */
/*           either be a string, such as "C:\lw\instr\file" or a pointer   */
/*           to such a string.  The return value is equal to the global    */
/*           error variable.                                               */
/*=========================================================================*/
int tg1304_read_data_file (instrID, filename)
int instrID;
char *filename;
{
    if (ibrdf (bd[instrID], filename) <= 0)
        tg1304_err = 229;
    else
        tg1304_err = 0;

    return tg1304_err;
}
/*=========================================================================*/
/* Function: Write Data File                                               */
/* Purpose:  This function writes a buffer of data from the file specified */
/*           by "filename" to the instrument. Filename must either be a    */
/*           string, such as "C:\lw\instr\file" or a pointer to such a     */
/*           string.  The return value is equal to the global error        */
/*           variable.                                                     */
/*=========================================================================*/
int tg1304_write_data_file (instrID, filename)
int instrID;
char *filename;
{
    if (ibwrtf (bd[instrID], filename) <= 0)
        tg1304_err = 228;
    else
        tg1304_err = 0;

    return tg1304_err;
}
/*=========================================================================*/
/* Function: Setup Arrays                                                  */
/* Purpose:  This function is called by the init routine to initialize     */
/*           global arrays.                                                */
/*=========================================================================*/
void tg1304_setup_arrays ()
{
    main_func[0] = "SIN";
    main_func[1] = "TRI";
    main_func[2] = "SQR";
    main_func[3] = "POSPUL";
    main_func[4] = "NEGPUL";
    main_func[5] = "POSPK";
    main_func[6] = "NEGPK";
    main_func[7] = "DCO";
    main_out[0] = "OFF";
    main_out[1] = "ON";

    aux_func[0] = "SIN";
    aux_func[1] = "TRI";
    aux_func[2] = "SQR";
    aux_func[3] = "RMP";
    aux_func[4] = "FCV";
    aux_func[5] = "ACV";
    aux_out[0] = "OFF";
    aux_out[1] = "ON";

    mod_enab[0] = "ON";
    mod_enab[1] = "OFF";
    mod_enab[2] = "CLEAR";
    mod_wave[0] = "SIN";
    mod_wave[1] = "TRI";
    mod_wave[2] = "SQR";
    mod_wave[3] = "EXT";

    vcaf_mode[0] = "INTLIN";
    vcaf_mode[1] = "INTLOG";
    vcaf_mode[2] = "EXTLIN";
    vcaf_mode[3] = "EXTLOG";

    set_status[0] = "*SRE";
    set_status[1] = "*ESE";
    set_status[2] = "*PRE";

    set_store[0] = "*RCL";
    set_store[1] = "*SAV";

    get_status[0] = "*STB?";
    get_status[1] = "*ESR?";
    get_status[2] = "*SRE?";
    get_status[3] = "*ESE?";
    get_status[4] = "*PRE?";

}
/*= THE END ===============================================================*/
